import { getEmailFromSAP } from "../services/sapService.js";
import { generateOTP, verifyOTP } from "../services/otpService.js";
import { sendOTPEmail } from "../services/emailService.js";
import jwt from "jsonwebtoken";
import {getPartnerCodeFromSAP, getUnitsByPartnerCode,getUserInfoFromSAP } from "../services/sapService.js";
import { getInstallmentsByContractNo } from "../services/sapService.js";
import axios from "axios";
import crypto from "crypto";


export const requestOTP = async (req, res) => {
  const { nationalId } = req.body;

  try {

        // If SKIP_OTP is enabled, bypass sending OTP for testing
    if ((process.env.SKIP_OTP || "").toLowerCase() === "true") {
      console.log(`SKIP_OTP enabled - skipping OTP send for ${nationalId}`);
      return res.json({ message: "OTP skipped (testing mode)" });
    }

    const email = await getEmailFromSAP(nationalId);
    if (!email) {
      return res.status(404).json({ message: "National ID not found" });
    }
    console.log(`✅ Email retrieved for ${nationalId}: ${email}`);

    const otp = generateOTP(nationalId);
    await sendOTPEmail(email, otp);

    return res.json({ message: "OTP sent to email" });
  } catch (err) {
    return res.status(500).json({ message: err.message });
  }
};

export const verifyOTPController = (req, res) => {

  const { nationalId, otp } = req.body;

    // If SKIP_OTP is enabled, accept any OTP and return a token
  if ((process.env.SKIP_OTP || "").toLowerCase() === "true") {
    console.log(`SKIP_OTP enabled - auto-login for ${nationalId}`);
    const token = jwt.sign({ nationalId }, process.env.JWT_SECRET, { expiresIn: "1h" });
    return res.json({ message: "Login successful (skipped OTP)", token });
  }

  if (verifyOTP(nationalId, otp)) {
    const token = jwt.sign({ nationalId }, process.env.JWT_SECRET, { expiresIn: "1h" });
    return res.json({ message: "Login successful", token });
  } else {
    return res.status(400).json({ message: "Invalid or expired OTP" });
  }
};


export const getUnitsController = async (req, res) => {
  res.setHeader('Cache-Control', 'no-store');

  try {
    const authHeader = req.headers.authorization;
    if (!authHeader) return res.status(401).json({ message: "Missing token" });
    const token = authHeader.split(" ")[1];
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const nationalId = decoded.nationalId;
    const partnerCode = await getPartnerCodeFromSAP(nationalId);
    if (!partnerCode) return res.status(404).json({ message: "Partner code not found" });
    const userInfo = await getUserInfoFromSAP(partnerCode);
    console.log("User info from SAP:", userInfo);
    const englishName = userInfo.englishName;
    const units = await getUnitsByPartnerCode(partnerCode);


        // normalize unit fields so frontend can always use `unit.project`
    const mappedUnits = (units || []).map(u => ({
      unit: u.unit || "",
      soldprice: u.soldprice || 0,
      contract: u.contract || "",
      installments: u.installments || [],
      // normalize possible project field names returned by SAP
      project: u.project || u.project_NAME || u.Project_NAME || u.Project || u.projectName || ""
    }));


    //return res.json({ units, englishName });
    return res.json({ units: mappedUnits, englishName });

  } catch (err) {
    return res.status(500).json({ message: err.message });
  }
};


export const getInstallmentsController = async (req, res) => {
  try {
    console.log("Installments requested for contract:", req.params.contractNo); // Debug: show contract number
    const { contractNo } = req.params;
    const installments = await getInstallmentsByContractNo(contractNo);
    console.log("Installments returned:", installments); // Debug: show returned installments

    res.json({ installments });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};


export const createPaymentSession = async (req, res) => {
  const authHeader = req.headers.authorization;
if (!authHeader) return res.status(401).json({ message: "Missing token" });

const token = authHeader.split(" ")[1];
const decoded = jwt.verify(token, process.env.JWT_SECRET);
const nationalId = decoded.nationalId;

// Fetch email automatically
const emailFromSAP = await getEmailFromSAP(nationalId);

  try {
    const {
      project,
      amount,
      currency = "EGP",
      orderId,
      description = "Test payment",
      customerEmail,
      customerPhone,
    } = req.body;

    if (!project || !amount || !orderId) {
  return res.status(400).json({ message: "project, amount, and orderId are required" });
}

    
    // inside createPaymentSession
const projectConfigs = {
  "LANUOVA VISTA": {
    merchantIdentifier: process.env.LANOUVA_VISTA_TEST_MERCHANT_IDENTIFIER,
    accessCode: process.env.LANOUVA_VISTA_TEST_ACCESS_CODE,
    shaRequestPhrase: process.env.LANOUVA_VISTA_TEST_SHA_REQUEST_PHRASE,
    shaResponsePhrase: process.env.LANOUVA_VISTA_TEST_SHA_RESPONSE_PHRASE,
  },
  "IL BOSCO": {
    merchantIdentifier: process.env.IL_BOSCO_TEST_MERCHANT_IDENTIFIER,
    accessCode: process.env.IL_BOSCO_TEST_ACCESS_CODE,
    shaRequestPhrase: process.env.IL_BOSCO_TEST_SHA_REQUEST_PHRASE,
    shaResponsePhrase: process.env.IL_BOSCO_TEST_SHA_RESPONSE_PHRASE,
  },
  "IL BOSCO CITY BM": {
    merchantIdentifier: process.env.IL_BOSCO_CITY_BM_TEST_MERCHANT_IDENTIFIER,
    accessCode: process.env.IL_BOSCO_CITY_BM_TEST_ACCESS_CODE,
    shaRequestPhrase: process.env.IL_BOSCO_CITY_BM_TEST_SHA_REQUEST_PHRASE,
    shaResponsePhrase: process.env.IL_BOSCO_CITY_BM_TEST_SHA_RESPONSE_PHRASE,
  },
  "SOLARE": {
    merchantIdentifier: process.env.SOLARE_TEST_MERCHANT_IDENTIFIER,
    accessCode: process.env.SOLARE_TEST_ACCESS_CODE,
    shaRequestPhrase: process.env.SOLARE_TEST_SHA_REQUEST_PHRASE,
    shaResponsePhrase: process.env.SOLARE_TEST_SHA_RESPONSE_PHRASE,
  },
  "KAI SOKHNA": {
    merchantIdentifier: process.env.KAI_TEST_MERCHANT_IDENTIFIER,
    accessCode: process.env.KAI_TEST_ACCESS_CODE,
    shaRequestPhrase: process.env.KAI_TEST_SHA_REQUEST_PHRASE,
    shaResponsePhrase: process.env.KAI_TEST_SHA_RESPONSE_PHRASE,
  },
};



    // Dynamically load credentials based on project
const projectKey = (project || "").trim().toUpperCase();
const config = projectConfigs[projectKey];

if (!config) {
  return res.status(400).json({ message: `Unsupported project: ${project}` });
}

const { merchantIdentifier, accessCode, shaRequestPhrase, shaResponsePhrase } = config;

if (!merchantIdentifier || !accessCode || !shaRequestPhrase || !shaResponsePhrase) {
  console.error(`Missing APS credentials for project ${projectKey}`);
  return res.status(500).json({ message: `Payment configuration missing for ${projectKey}` });
}

    // --- Convert amount to smallest unit (piasters) ---
    const amountValue = Math.round(Number(amount * 100));

const payload = {
  service_command: "PAYMENT_LINK",
  access_code: accessCode,
  merchant_identifier: merchantIdentifier,
  merchant_reference: orderId,
  notification_type:"EMAIL",
  order_description: description || `Installment payment for ${project}`,
  amount: String(amountValue),
  currency: currency,
  language: "en",
  request_expiry_date: "2026-06-12T12:38:00+03:00",
  customer_email: emailFromSAP || customerEmail,
  customer_phone: "201147874765",
  settlement_reference: `SETTLEMENT-${Date.now()}`,
  return_url: process.env.PAY_RETURN_URL || "https://example.com/payment/return"
};


    const signingParams = Object.keys(payload)
      .filter((k) => payload[k] !== undefined && payload[k] !== null && payload[k] !== "")
      .sort()
      .map((k) => `${k}=${payload[k]}`)
      .join("");

    const signString = shaRequestPhrase + signingParams + shaRequestPhrase;
    const signature = crypto.createHash("sha256").update(signString, "utf8").digest("hex");

    payload.signature = signature;

    console.log("🧾 PayFort signString:", signString);
    console.log("🧾 Calculated signature:", signature);

    const url = "https://sbpaymentservices.payfort.com/FortAPI/paymentApi/v1.1/paymentApi";
    const headers = {
      "Content-Type": "application/json;charset=UTF-8",
      Connection: "keep-alive",
    };

    const response = await axios.post(url, payload, { headers });
    console.log("✅ PayFort payment link response:", response.status, response.data);
    const respBody = response.data || {};

    const filtered = Object.entries(respBody)
      .filter(([k, v]) => k !== "signature" && v !== undefined && v !== null && v !== "")
      .sort(([a], [b]) => a.localeCompare(b));

    const signingStringResp =
      shaResponsePhrase +
      filtered.map(([k, v]) => `${k}=${String(v).trim()}`).join("") +
      shaResponsePhrase;

    const calculatedSignature = crypto
      .createHash("sha256")
      .update(signingStringResp, "utf8")
      .digest("hex")
      .toLowerCase();

    if (calculatedSignature !== (respBody.signature || "").toLowerCase()) {
      console.warn("⚠️ Response signature mismatch");
    } else {
      console.log("✅ Response signature verified successfully");
    }

    const redirectUrl =
      respBody.payment_link || respBody.hosted_payment_url || respBody.redirect_url || null;

    if (!redirectUrl) {
      return res
        .status(400)
        .json({ message: "Payment link not returned", payResponse: respBody });
    }

    return res.json({
      success: true,
      redirectUrl,
      payResponse: respBody,
    });
  } catch (err) {
    console.error("❌ createPaymentSession error:", err.response?.data || err.message);
    return res.status(500).json({
      message: "Payment initiation failed",
      error: err.response?.data || err.message,
    });
  }
};